/*
 * Copyright (C) 2008 Robert Fitzsimons
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  Look for the COPYING file in the top
 * level directory.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.cyclerecorder.footprintbuilder.renderer;

import java.io.IOException;
import java.util.ArrayList;
import org.cyclerecorder.footprintbuilder.Utility;
import org.cyclerecorder.footprintbuilder.data.Footprint;
import org.cyclerecorder.footprintbuilder.data.PadPin;
import org.cyclerecorder.footprintbuilder.data.SilkLine;
import org.cyclerecorder.footprintbuilder.data.UnitType;

public final class PcbRenderer extends AppendableRenderer {
	public void render(final Appendable buffer, final Footprint footprint) throws IOException {
		final ArrayList<PadPin> padPins = footprint.getPadPins();
		final ArrayList<SilkLine> silkLines = footprint.getSilkLines();

		renderStart(buffer, footprint);
		for (final PadPin padPin : padPins) {
			renderPadPin(buffer, footprint, padPin);
		}
		for (final SilkLine silkLine : silkLines) {
			renderSilkLine(buffer, footprint, silkLine);
		}
		renderEnd(buffer, footprint);
	}

	private static void renderStart(final Appendable buffer, final Footprint footprint) throws IOException {
		buffer.append("Element[\"\"");
		buffer.append(" \"").append(toString(footprint.getDescription())).append("\"");
		buffer.append(" \"").append(toString(footprint.getName())).append("\"");
		buffer.append(" \"\" 0 0 0 0 0 100 \"\"]\n(\n");
	}

	private static void renderEnd(final Appendable buffer, final Footprint footprint) throws IOException {
		buffer.append(")\n");
	}

	private static void renderPadPin(final Appendable buffer, final Footprint footprint, final PadPin padPin) throws IOException {
		final double x = padPin.getX();
		final double y = padPin.getY();
		final String number = padPin.getNumber();

		if (padPin.isPad()) {
			final double xo = padPin.getXOffset();
			final double yo = padPin.getYOffset();
			final double width = padPin.getPadWidth();
			final double length = padPin.getPadLength();
			final double angle = padPin.getAngle();
			final boolean square = padPin.isRectangle();
			final double halfWidth = (width / 2.0D);
			final double halfLength = (length / 2.0D);

			double x1;
			double y1;
			double x2;
			double y2;
			double thickness;

			if (width < length) {
				thickness = width;
				x1 = x2 = 0.0D;
				y1 = -halfLength + halfWidth;
				y2 = halfLength - halfWidth;
			} else {
				thickness = length;
				x1 = -halfWidth + halfLength;
				x2 = halfWidth - halfLength;
				y1 = y2 = 0.0D;
			}

			x1 += xo;
			y1 += yo;
			x2 += xo;
			y2 += yo;

			final double clearance = (padPin.getClearanceOffset() * 2.0D);
			final double mask = (thickness + (padPin.getMaskOffset() * 2.0D));

			final double cos = Utility.getCosOfDegrees(angle);
			final double sin = Utility.getSinOfDegrees(angle);
			final double tx1 = ((cos * x1) + (-sin * y1)) + x;
			final double ty1 = ((sin * x1) + (cos * y1)) + y;
			final double tx2 = ((cos * x2) + (-sin * y2)) + x;
			final double ty2 = ((sin * x2) + (cos * y2)) + y;

			renderPad(buffer, footprint.getUnitType(), tx1, ty1, tx2, ty2, thickness, clearance, mask, number, square);
		}
		if (padPin.isPin() || padPin.isHoleOnly()) {
			final double thickness = padPin.getPinDiameter();
			final double drill = padPin.getHoleDiameter();
			final boolean holeOnly = padPin.isHoleOnly();

			final double clearance = (padPin.getClearanceOffset() * 2.0D);
			final double mask = (Math.max(thickness, drill) + (padPin.getMaskOffset() * 2.0D));

			renderPin(buffer, footprint.getUnitType(), x, y, thickness, clearance, mask, drill, number, holeOnly);
		}
	}

	private static void renderSilkLine(final Appendable buffer, final Footprint footprint, final SilkLine silkLine) throws IOException {
		final double x1 = silkLine.getX1();
		final double y1 = silkLine.getY1();
		final double x2 = silkLine.getX2();
		final double y2 = silkLine.getY2();
		final double thickness = silkLine.getThickness();

		renderLine(buffer, footprint.getUnitType(), x1, y1, x2, y2, thickness);
	}

	private static void renderPad(final Appendable buffer, final UnitType unitType, final double x1, final double y1, final double x2, final double y2, final double thickness, final double clearance, final double mask, final String number, final boolean square) throws IOException {
		buffer.append("\tPad[").append(toCentMil(unitType, x1));
		buffer.append(" ").append(toCentMil(unitType, y1));
		buffer.append(" ").append(toCentMil(unitType, x2));
		buffer.append(" ").append(toCentMil(unitType, y2));
		buffer.append(" ").append(toCentMil(unitType, thickness));
		buffer.append(" ").append(toCentMil(unitType, clearance));
		buffer.append(" ").append(toCentMil(unitType, mask));
		buffer.append(" \"").append(toString(number)).append("\"");
		buffer.append(" \"").append(toString(number)).append("\"");
		buffer.append(" \"").append((square) ? ("square") : ("")).append("\"");
		buffer.append("]\n");
	}

	private static void renderPin(final Appendable buffer, final UnitType unitType, final double x, final double y, final double thickness, final double clearance, final double mask, final double drill, final String number, final boolean holeOnly) throws IOException {
		buffer.append("\tPin[").append(toCentMil(unitType, x));
		buffer.append(" ").append(toCentMil(unitType, y));
		buffer.append(" ").append(toCentMil(unitType, thickness));
		buffer.append(" ").append(toCentMil(unitType, clearance));
		buffer.append(" ").append(toCentMil(unitType, mask));
		buffer.append(" ").append(toCentMil(unitType, drill));
		buffer.append(" \"").append(toString(number)).append("\"");
		buffer.append(" \"").append(toString(number)).append("\"");
		buffer.append(" \"").append((holeOnly) ? ("hole") : ("")).append("\"");
		buffer.append("]\n");
	}

	public static void renderLine(final Appendable buffer, final UnitType unitType, final double x1, final double y1, final double x2, final double y2, final double thickness) throws IOException {
		buffer.append("\tElementLine[").append(toCentMil(unitType, x1));
		buffer.append(" ").append(toCentMil(unitType, y1));
		buffer.append(" ").append(toCentMil(unitType, x2));
		buffer.append(" ").append(toCentMil(unitType, y2));
		buffer.append(" ").append(toCentMil(unitType, thickness));
		buffer.append("]\n");
	}

	private static String toCentMil(final UnitType unit, final double value) {
		return Integer.toString((int)unit.toCentiMil(value));
	}

	private static String toString(final String string) {
		return ((string != null) ? (string) : (""));
	}
}

