/*
 * Copyright (C) 2008, 2009 Robert Fitzsimons
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  Look for the COPYING file in the top
 * level directory.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.cyclerecorder.footprintbuilder.renderer;

import java.io.IOException;
import java.util.ArrayList;
import org.cyclerecorder.footprintbuilder.data.Footprint;
import org.cyclerecorder.footprintbuilder.data.PadPin;
import org.cyclerecorder.footprintbuilder.data.SilkLine;
import org.cyclerecorder.footprintbuilder.data.UnitType;

public final class KicadRenderer extends AppendableRenderer {
	public void render(final Appendable buffer, final Footprint footprint) throws IOException {
		final ArrayList<PadPin> padPins = footprint.getPadPins();
		final ArrayList<SilkLine> silkLines = footprint.getSilkLines();
		final UnitType unitType = footprint.getUnitType();

		renderStart(buffer, footprint);
		for (final SilkLine silkLine : silkLines) {
			final double x1 = silkLine.getX1();
			final double y1 = silkLine.getY1();
			final double x2 = silkLine.getX2();
			final double y2 = silkLine.getY2();
			final double thickness = silkLine.getThickness();

			buffer.append("DS ").append(toDeciMil(unitType, x1));
			buffer.append(" ").append(toDeciMil(unitType, y1));
			buffer.append(" ").append(toDeciMil(unitType, x2));
			buffer.append(" ").append(toDeciMil(unitType, y2));
			buffer.append(" ").append(toDeciMil(unitType, thickness));
			buffer.append(" 21\n");
		}
		for (final PadPin padPin : padPins) {
			final String number = padPin.getNumber();
			final double x = padPin.getX();
			final double px = padPin.getXOffset();
			final double y = padPin.getY();
			final double py = padPin.getYOffset();
			final double width = padPin.getPadWidth();
			final double length = padPin.getPadLength();
			final boolean rectangle = padPin.isRectangle();
			final double pin = padPin.getPinDiameter();
			final double hole = padPin.getHoleDiameter();
			final boolean holeOnly = padPin.isHoleOnly();

			double angle = padPin.getAngle();
			if (angle == 90.0D) {
				angle = 270.0D;
			} else if (angle == 270.0D) {
				angle = 90.0D;
			}
			angle *= 10.0D;

			double w = width;
			double l = length;
			String shape = ((rectangle) ? ("R") : ("O"));

			if (padPin.isHoleOnly()) {
				w = l = hole;
				shape = "C";
			} else if (padPin.isPin()) {
				w = l = pin;
				shape = "C";
			}

			buffer.append("$PAD\n");
			buffer.append("Sh \"").append(toString(number)).append("\"");
			buffer.append(" ").append(shape);
			buffer.append(" ").append(toDeciMil(unitType, w));
			buffer.append(" ").append(toDeciMil(unitType, l));
			buffer.append(" ").append(toDeciMil(unitType, 0.0D));
			buffer.append(" ").append(toDeciMil(unitType, 0.0D));
			buffer.append(" ").append(Integer.toString((int)angle)).append("\n");

			buffer.append("Dr ").append(toDeciMil(unitType, hole));
			buffer.append(" ").append(toDeciMil(unitType, px));
			buffer.append(" ").append(toDeciMil(unitType, py)).append("\n");

			if (padPin.isPad()) {
				buffer.append("At SMD N 00888000\n");
			} else if (padPin.isHoleOnly()) {
				buffer.append("At HOLE N 00F0FFFF\n");
			} else if (padPin.isPin()) {
				buffer.append("At STD N 00E0FFFF\n");
			}

			buffer.append("Po ").append(toDeciMil(unitType, x));
			buffer.append(" ").append(toDeciMil(unitType, y)).append("\n");
			buffer.append("$EndPAD\n");
		}
		renderEnd(buffer, footprint);
	}

	private static void renderStart(final Appendable buffer, final Footprint footprint) throws IOException {
		buffer.append("PCBNEW-LibModule-V1\n");
		buffer.append("$INDEX\n");
		buffer.append(toString(footprint.getName())).append("\n");
		buffer.append("$EndINDEX\n");
		buffer.append("$MODULE ").append(toString(footprint.getName())).append("\n");
		buffer.append("Po 0 0 0 15 0 0 ~~\n");
		buffer.append("Li ").append(toString(footprint.getName())).append("\n");
		if (toString(footprint.getDescription()).length() > 0) {
			buffer.append("Cd ").append(toString(footprint.getDescription())).append("\n");
		}
		buffer.append("Sc 0\n");
		buffer.append("Op 0 0 0\n");
		buffer.append("T0 0 0 600 600 0 120 N V 21 \"").append(toString(footprint.getName())).append("\"\n");
		buffer.append("T1 0 0 600 600 0 120 N V 21 \"VAL**\"\n");
	}

	private static void renderEnd(final Appendable buffer, final Footprint footprint) throws IOException {
		buffer.append("$EndMODULE ").append(toString(footprint.getName())).append("\n");
		buffer.append("$EndLIBRARY\n");
	}

	private static String toDeciMil(final UnitType unit, final double value) {
		return Integer.toString((int)unit.toDeciMil(value));
	}

	private static String toString(final String string) {
		return ((string != null) ? (string) : (""));
	}
}

