/*
 * Copyright (C) 2008 Robert Fitzsimons
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  Look for the COPYING file in the top
 * level directory.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.cyclerecorder.footprintbuilder.renderer;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import java.awt.geom.RoundRectangle2D;
import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumMap;
import org.cyclerecorder.footprintbuilder.data.BoundingRectangle;
import org.cyclerecorder.footprintbuilder.data.Footprint;
import org.cyclerecorder.footprintbuilder.data.Group;
import org.cyclerecorder.footprintbuilder.data.PadPin;
import org.cyclerecorder.footprintbuilder.data.SilkLine;

public final class Graphics2DRenderer extends Renderer<Graphics2D> {
	private static final Color boardColor = new Color(0xCC, 0xCC, 0x77);
	private static final Color copperColor = new Color(0x66, 0x66, 0x66);
	private static final Color maskColor = new Color(0x00, 0x99, 0x00);
	private static final Color clearColor = new Color(0xCC, 0xCC, 0xCC);
	private static final Color silkColor = Color.WHITE;
	private static final Color outlineColor = Color.BLACK;
	private static final Color numberColor = new Color(0xFF, 0x00, 0x00);
	private static final Color overlayColor = Color.BLACK;

	public void render(final Graphics2D g2, final Footprint footprint) throws IOException {
		final EnumMap<Group, BoundingRectangle> groupBounds = footprint.buildGroupBounds(0.0D);
		final BoundingRectangle allBR = groupBounds.get(Group.ALL);

		final Rectangle bounds = ((g2.getClipBounds() != null) ? (g2.getClipBounds()) : (g2.getDeviceConfiguration().getBounds()));
		final int boundsWidth = (bounds.x + bounds.width);
		final int boundsHeight = (bounds.y + bounds.height);

		g2.setColor(maskColor);
		g2.fillRect(0, 0, boundsWidth, boundsHeight);

		g2.translate((boundsWidth / 2), (boundsHeight / 2));

		final AffineTransform tx = g2.getTransform();

		{
			final double scaleW = (((double)boundsWidth) / (allBR.getWidth() * 1.5D));
			final double scaleL = (((double)boundsHeight) / (allBR.getLength() * 1.5D));
			final double scale = Math.min(scaleW, scaleL);
			g2.scale(scale, scale);
		}

		this.renderFootprint(g2, footprint);

		g2.setTransform(tx);

		g2.setColor(overlayColor);
		g2.setFont(new Font(Font.MONOSPACED, Font.PLAIN, 12));

		g2.drawLine(-(boundsWidth / 2), 0, (boundsWidth / 2), 0);
		g2.drawLine(0, -(boundsHeight / 2), 0, (boundsHeight / 2));

		drawText(g2, "Top View", 0, ((-boundsHeight / 2) + (g2.getFontMetrics().getHeight() * 1)));
	}

	private void renderFootprint(final Graphics2D g2, final Footprint footprint) throws IOException {
		final ArrayList<PadPin> padPins = footprint.getPadPins();
		final ArrayList<SilkLine> silkLines = footprint.getSilkLines();

		final ArrayList<Shape> negMaskList = new ArrayList<Shape>();
		final ArrayList<Shape> copperList = new ArrayList<Shape>();
		final ArrayList<Shape> negHoleList = new ArrayList<Shape>();
		final ArrayList<Shape> outlineList = new ArrayList<Shape>();

		for (final PadPin padPin : padPins) {
			final double x = padPin.getX();
			final double y = padPin.getY();

			if (padPin.isPad()) {
				final double xo = padPin.getXOffset();
				final double yo = padPin.getYOffset();
				final double width = padPin.getPadWidth();
				final double length = padPin.getPadLength();
				final double angle = padPin.getAngle();
				final boolean square = padPin.isRectangle();
				final double mask = (padPin.getMaskOffset() * 2.0D);

				copperList.add(createShape(x, y, xo, yo, angle, width, length, square));
				negMaskList.add(createShape(x, y, xo, yo, angle, (width + mask), (length + mask), square));
			}
			if (padPin.isPin() || padPin.isHoleOnly()) {
				final double pinDiameter = padPin.getPinDiameter();
				final double holeDiameter = padPin.getHoleDiameter();
				final double mask = (padPin.getMaskOffset() * 2.0D);

				if (!padPin.isHoleOnly()) {
					copperList.add(createShape(x, y, pinDiameter, false));
					negMaskList.add(createShape(x, y, (pinDiameter + mask), false));
				}
				negHoleList.add(createShape(x, y, holeDiameter, false));
			}
		}

		g2.setColor(boardColor);
		for (final Shape maskShape : negMaskList) {
			g2.fill(maskShape);
		}
		g2.setColor(copperColor);
		for (final Shape copperShape : copperList) {
			g2.fill(copperShape);
		}
		g2.setColor(clearColor);
		for (final Shape holeShape : negHoleList) {
			g2.fill(holeShape);
		}

		g2.setColor(silkColor);
		final Stroke orignalStroke = g2.getStroke();
		for (final SilkLine silkLine : silkLines) {
			final Line2D.Double line = new Line2D.Double(silkLine.getX1(), silkLine.getY1(), silkLine.getX2(), silkLine.getY2());
			g2.setStroke(new BasicStroke((float)silkLine.getThickness(), BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g2.draw(line);
		}
		g2.setStroke(orignalStroke);

		g2.setColor(outlineColor);
		for (final Shape outlineShape : outlineList) {
			g2.draw(outlineShape);
		}
	}

	private static Shape createShape(final double x, final double y, final double xo, final double yo, final double angle, final double width, final double length, final boolean square) {
		final double halfWidth = (width / 2.0D);
		final double halfLength = (length / 2.0D);

		Shape shape;
		if (square) {
			shape = new Rectangle2D.Double(-halfWidth, -halfLength, width, length);
		} else {
			final double arc = Math.min(width, length);
			shape = new RoundRectangle2D.Double(-halfWidth, -halfLength, width, length, arc, arc);
		}

		shape = AffineTransform.getTranslateInstance(xo, yo).createTransformedShape(shape);
		shape = AffineTransform.getRotateInstance(Math.toRadians(angle)).createTransformedShape(shape);
		shape = AffineTransform.getTranslateInstance(x, y).createTransformedShape(shape);
		return shape;
	}

	private static Shape createShape(final double x, final double y, final double side, final boolean square) {
		final double halfSide = (side / 2.0D);

		if (square) {
			return new Rectangle2D.Double((x + -halfSide), (y + -halfSide), side, side);
		} else {
			return new Ellipse2D.Double((x + -halfSide), (y + -halfSide), side, side);
		}
	}

	private static void drawText(final Graphics2D g2, final String string, final int x, final int y) {
		final FontMetrics fontMetrics = g2.getFontMetrics();
		final int width = fontMetrics.stringWidth(string);
		g2.drawString(string, (x - (width / 2)), y);
	}
}

