/*
 * Copyright (C) 2008 Robert Fitzsimons
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  Look for the COPYING file in the top
 * level directory.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.cyclerecorder.footprintbuilder.gui;

import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeListener;
import org.cyclerecorder.footprintbuilder.editable.EditableDouble;
import org.cyclerecorder.footprintbuilder.editable.EditableInteger;
import org.cyclerecorder.footprintbuilder.editable.EditableNumber;
import org.cyclerecorder.footprintbuilder.data.UnitType;

public final class EditableJSpinner extends JSpinner {
	private final boolean isPositive;

	private EditableJSpinner(final boolean isPositive, final EditableSpinnerNumberModel model, final ChangeListener changeListener) {
		super(model);
		this.isPositive = isPositive;
		this.addChangeListener(changeListener);
	}
	public EditableJSpinner(final boolean isPositive, final EditableInteger editable, final ChangeListener changeListener) {
		this(
			isPositive,
			((isPositive) ? (new EditableSpinnerNumberModel(editable, 0, 100, 1)) : (new EditableSpinnerNumberModel(editable, -100, 100, 1))),
 			changeListener
		);
	}

	public EditableJSpinner(final boolean isPositive, final EditableDouble editable, final ChangeListener changeListener) {
		this(isPositive, new EditableSpinnerNumberModel(editable, 0.0D, 1000.0000D, 0.0D), changeListener);
	}

	private UnitType unitType;
	public void setUnitType(final UnitType unitType) {
		final UnitType oldUnitType = this.unitType;

		if (unitType == oldUnitType) {
			return;
		}

		this.unitType = unitType;

		final EditableSpinnerNumberModel model = (EditableSpinnerNumberModel)this.getModel();
		if (!(model.editable instanceof EditableDouble)) {
			throw new UnsupportedOperationException("The setUnitType method is only supported for EditableDouble objects.");
		}

		double minimum = 0.0D;
		double maximum = 0.0D;
		double stepSize = 0.0D;

		switch (this.unitType) {
		case MM:
			maximum = 200.0D;
			minimum = -maximum;
			stepSize = 0.01D;
			break;
		case MIL:
			maximum = 8000.0D;
			minimum = -maximum;
			stepSize = 0.5D;
			break;
		case INCH:
			maximum = 8.0D;
			minimum = -maximum;
			stepSize = 0.001D;
			break;
		}
		if (this.isPositive) {
			minimum = 0.0D;
		}

		model.setMinimum(Double.valueOf(minimum));
		model.setMaximum(Double.valueOf(maximum));
		model.setStepSize(Double.valueOf(stepSize));

		double value = UnitType.convert(oldUnitType, model.editable.doubleValue(), this.unitType);
		value = (((long)(value / stepSize)) * stepSize);

		model.setValue(Double.valueOf(value));
	}

	private final static class EditableSpinnerNumberModel extends SpinnerNumberModel {
		private final EditableNumber editable;

		public EditableSpinnerNumberModel(final EditableInteger editable, final int minimum, final int maximum, final int stepSize) {
			super(editable.intValue(), minimum, maximum, stepSize);
			this.editable = editable;
		}
		public EditableSpinnerNumberModel(final EditableDouble editable, final double minimum, final double maximum, final double stepSize) {
			super(editable.doubleValue(), minimum, maximum, stepSize);
			this.editable = editable;
		}

		public void setValue(final Object value) {
			if (value instanceof Number) {
				this.editable.setValue((Number)value);
			}
			super.setValue(value);
		}
	}
}

