/*
 * Copyright (C) 2008 Robert Fitzsimons
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  Look for the COPYING file in the top
 * level directory.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.cyclerecorder.footprintbuilder.editable;

import org.cyclerecorder.footprintbuilder.editable.EditableEnum;
import org.cyclerecorder.footprintbuilder.editable.EditableDouble;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.IdentityHashMap;
import org.cyclerecorder.footprintbuilder.data.BoundingRectangle;
import org.cyclerecorder.footprintbuilder.data.Footprint;
import org.cyclerecorder.footprintbuilder.data.Group;
import org.cyclerecorder.footprintbuilder.data.Line;
import org.cyclerecorder.footprintbuilder.data.SegmentLine;
import org.cyclerecorder.footprintbuilder.data.SilkLine;
import org.cyclerecorder.footprintbuilder.data.SilkMarkerType;
import org.cyclerecorder.footprintbuilder.data.StyleType;

public class SilkEditable {
	public boolean isValid() {
		return ((this.getSilkOffset() > 0.0D) || ((this.getSilkWidth() > 0.0D) && (this.getSilkLength() > 0.0D)));
	}

	private StyleType silkStyle = StyleType.RECTANGLE;
	public StyleType getSilkStyle() {
		return this.silkStyle;
	}
	public void setSilkStyle(final StyleType silkStyle) {
		this.silkStyle = silkStyle;
	}

	private final EditableEnum<SilkMarkerType> silkMarker = new EditableEnum<SilkMarkerType>(SilkMarkerType.values(), SilkMarkerType.NONE);
	public SilkMarkerType getSilkMarker() {
		return this.silkMarker.getValue();
	}
	public EditableEnum<SilkMarkerType> getSilkMarkerEditable() {
		return this.silkMarker;
	}

	private final EditableDouble silkOffset = new EditableDouble();
	public double getSilkOffset() {
		return this.silkOffset.doubleValue();
	}
	public EditableDouble getSilkOffsetEditable() {
		return this.silkOffset;
	}

	private final EditableDouble silkWidth = new EditableDouble();
	public double getSilkWidth() {
		return this.silkWidth.doubleValue();
	}
	public EditableDouble getSilkWidthEditable() {
		return this.silkWidth;
	}

	private final EditableDouble silkLength = new EditableDouble();
	public double getSilkLength() {
		return this.silkLength.doubleValue();
	}
	public EditableDouble getSilkLengthEditable() {
		return this.silkLength;
	}

	private final EditableDouble silkX = new EditableDouble();
	public double getSilkX() {
		return this.silkX.doubleValue();
	}
	public EditableDouble getSilkXEditable() {
		return this.silkX;
	}

	private final EditableDouble silkY = new EditableDouble();
	public double getSilkY() {
		return this.silkY.doubleValue();
	}
	public EditableDouble getSilkYEditable() {
		return this.silkY;
	}

	public void fillFootprint(final Footprint footprint) {
		if (!this.isValid()) {
			return;
		}

		final EnumMap<Group, BoundingRectangle> groupBounds = footprint.buildGroupBounds(footprint.getOptions().getSilkLineWidth() / 2.0D);
		final ArrayList<SilkLine> newSilkLines = new ArrayList<SilkLine>();
		final ArrayList<BoundingRectangle> bounds = new ArrayList<BoundingRectangle>();

		this.drawSilk(footprint, newSilkLines, bounds, groupBounds);

		for (final SilkLine silkLine : newSilkLines) {
			final SegmentLine segmentLine = new SegmentLine(silkLine.getX1(), silkLine.getY1(), silkLine.getX2(), silkLine.getY2());
			segmentLine.splitLine(bounds);
			for (final Line line : segmentLine.getLines()) {
				footprint.addSilkLine(new SilkLine(line.getX1(), line.getY1(), line.getX2(), line.getY2(), silkLine.getThickness()));
			}
		}
	}

	private void drawSilk(final Footprint footprint, final ArrayList<SilkLine> silkLines, final ArrayList<BoundingRectangle> bounds, final EnumMap<Group, BoundingRectangle> groupBounds) {
		final double lineWidth = footprint.getOptions().getSilkLineWidth();

		double el;
		double er;
		double et;
		double eb;

		if (this.getSilkOffset() > 0.0D) {
			final double silkOffset = this.getSilkOffset();
			final BoundingRectangle allBR = groupBounds.get(Group.ALL);
			el = allBR.getLeft() + -silkOffset;
			er = allBR.getRight() + silkOffset;
			et = allBR.getTop() + -silkOffset;
			eb = allBR.getBottom() + silkOffset;
		} else {
			final double halfWidth = (this.getSilkWidth() / 2.0D);
			final double halfLength = (this.getSilkLength() / 2.0D);
			final double x = this.getSilkX();
			final double y = this.getSilkY();
			el = -halfWidth + x;
			er = halfWidth + x;
			et = -halfLength + y;
			eb = halfLength + y;
		}

		final double markerOffset = (lineWidth * 3.0D);

		silkLines.add(new SilkLine(el, et, er, et, lineWidth));
		silkLines.add(new SilkLine(er, et, er, eb, lineWidth));
		silkLines.add(new SilkLine(er, eb, el, eb, lineWidth));
		silkLines.add(new SilkLine(el, eb, el, et, lineWidth));

		switch (this.getSilkMarker()) {
		case NONE: break;
		case BEVELED: silkLines.add(new SilkLine(el, eb - markerOffset, el + markerOffset, eb, lineWidth)); break;
		case SQUARE:
			silkLines.add(new SilkLine(el, eb - markerOffset, el + markerOffset, eb - markerOffset, lineWidth));
			silkLines.add(new SilkLine(el + markerOffset, eb - markerOffset, el + markerOffset, eb, lineWidth));
			break;
		case CUP:
			silkLines.add(new SilkLine(el, 0.0D + markerOffset, el + markerOffset, 0.0D + markerOffset, lineWidth));
			silkLines.add(new SilkLine(el + markerOffset, 0.0D + markerOffset, el + markerOffset, 0.0D - markerOffset, lineWidth));
			silkLines.add(new SilkLine(el, 0.0D - markerOffset, el + markerOffset, 0.0D - markerOffset, lineWidth));
			break;
		case BAND:
			{
				final BoundingRectangle pinOneBR = groupBounds.get(Group.PINONE);
				final double top = pinOneBR.getTop();
				silkLines.add(new SilkLine(el, top, er, top, lineWidth));
			}
			break;
		case AROUNDPINONE:
			{
				final BoundingRectangle pinOneBR = groupBounds.get(Group.PINONE);
				final double left = pinOneBR.getLeft();
				final double right = pinOneBR.getRight();
				final double top = pinOneBR.getTop();
				final double bottom = pinOneBR.getBottom();
				silkLines.add(new SilkLine(left, top, right, top, lineWidth));
				silkLines.add(new SilkLine(right, top, right, bottom, lineWidth));
				silkLines.add(new SilkLine(right, bottom, left, bottom, lineWidth));
				silkLines.add(new SilkLine(left, bottom, left, top, lineWidth));
			}
			break;
		case INSIDE_DOT: silkLines.add(new SilkLine(el + markerOffset, eb - markerOffset, el + markerOffset, eb - markerOffset, (lineWidth * 2.0D))); break;
		case OUTSIDE_DOT: silkLines.add(new SilkLine(el - markerOffset, eb + markerOffset, el - markerOffset, eb + markerOffset, (lineWidth * 2.0D))); break;
		case INSIDE_TICK: silkLines.add(new SilkLine(el, eb, el + markerOffset, eb - markerOffset, lineWidth)); break;
		case OUTSIDE_TICK: silkLines.add(new SilkLine(el, eb, el - markerOffset, eb + markerOffset, lineWidth)); break;
		case INSIDE_LINE: silkLines.add(new SilkLine(el, eb - markerOffset, er, eb - markerOffset, lineWidth)); break;
		case OUTSIDE_LINE: silkLines.add(new SilkLine(el, eb + markerOffset, er, eb + markerOffset, lineWidth)); break;
		}

		for (final Group group : new Group[] {Group.TOP, Group.BOTTOM, Group.LEFT, Group.RIGHT, Group.EXPOSEDPAD, Group.GRID}) {
			final BoundingRectangle br = groupBounds.get(group);
			if (br != null) {
				bounds.add(br);
			}
		}
	}
}

