/*
 * Copyright (C) 2008 Robert Fitzsimons
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  Look for the COPYING file in the top
 * level directory.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.cyclerecorder.footprintbuilder.editable;

import org.cyclerecorder.footprintbuilder.data.Footprint;
import org.cyclerecorder.footprintbuilder.data.Group;
import org.cyclerecorder.footprintbuilder.data.OffsetType;
import org.cyclerecorder.footprintbuilder.data.PadPin;
import org.cyclerecorder.footprintbuilder.data.StyleType;
import org.cyclerecorder.footprintbuilder.editable.EditableEnum;
import org.cyclerecorder.footprintbuilder.editable.EditableInteger;
import org.cyclerecorder.footprintbuilder.editable.EditableDouble;

public class RectangleEditable {
	public abstract class SideEditable {
		private final PadEditable pad = new PadEditable();
		public PadEditable getPad() {
			return this.pad;
		}

		private final EditableInteger count = new EditableInteger();
		public int getCount() {
			return this.count.intValue();
		}
		public EditableInteger getCountEditable() {
			return this.count;
		}

		private final EditableDouble pitch = new EditableDouble();
		public double getPitch() {
			return this.pitch.doubleValue();
		}
		public EditableDouble getPitchEditable() {
			return this.pitch;
		}

		private final EditableEnum<OffsetType> offsetType = new EditableEnum<OffsetType>(OffsetType.values(), OffsetType.CENTRED);
		public OffsetType getOffsetType() {
			return this.offsetType.getValue();
		}
		public EditableEnum<OffsetType> getOffsetTypeEditable() {
			return this.offsetType;
		}

		protected abstract void fillFootprint(final Footprint footprint, final double halfOffset);
	}

	private final SideEditable bSide = new SideEditable() {
		protected void fillFootprint(final Footprint footprint, final double halfOffset) {
			final int count = this.getCount();
			final double pitch = this.getPitch();

			double x = -((pitch * (count - 1)) / 2.0D);
			final double y = halfOffset;
			for (int i = 0; i < count; i++) {
				final PadPin padPin = new PadPin(x, y, getNextNumber());
				fillPadPin(padPin, this.getPad(), 180.0D, this.getOffsetType(), footprint.getOptions());

				footprint.addGroup(Group.ALL, padPin);
				footprint.addGroup(Group.BOTTOM, padPin);

				x += pitch;
			}
		}
	};
	public SideEditable getBSide() {
		return this.bSide;
	}

	private final SideEditable rSide = new SideEditable() {
		protected void fillFootprint(final Footprint footprint, final double halfOffset) {
			final int count = this.getCount();
			final double pitch = this.getPitch();

			final double x = halfOffset;
			double y = ((pitch * (count - 1)) / 2.0D);
			for (int i = 0; i < count; i++) {
				final PadPin padPin = new PadPin(x, y, getNextNumber());
				fillPadPin(padPin, this.getPad(), 90.0D, this.getOffsetType(), footprint.getOptions());

				footprint.addGroup(Group.ALL, padPin);
				footprint.addGroup(Group.RIGHT, padPin);

				y -= pitch;
			}
		}
	};
	public SideEditable getRSide() {
		return this.rSide;
	}

	private final SideEditable tSide = new SideEditable() {
		protected void fillFootprint(final Footprint footprint, final double halfOffset) {
			final int count = this.getCount();
			final double pitch = this.getPitch();

			double x = ((pitch * (count - 1)) / 2.0D);
			final double y = -halfOffset;
			for (int i = 0; i < count; i++) {
				final PadPin padPin = new PadPin(x, y, getNextNumber());
				fillPadPin(padPin, this.getPad(), 0.0D, this.getOffsetType(), footprint.getOptions());

				footprint.addGroup(Group.ALL, padPin);
				footprint.addGroup(Group.TOP, padPin);

				x -= pitch;
			}
		}
	};
	public SideEditable getTSide() {
		return this.tSide;
	}

	private final SideEditable lSide = new SideEditable() {
		protected void fillFootprint(final Footprint footprint, final double halfOffset) {
			final int count = this.getCount();
			final double pitch = this.getPitch();

			final double x = -halfOffset;
			double y = -((pitch * (count - 1)) / 2.0D);
			for (int i = 0; i < count; i++) {
				final PadPin padPin = new PadPin(x, y, getNextNumber());
				fillPadPin(padPin, this.getPad(), 270.0D, this.getOffsetType(), footprint.getOptions());

				footprint.addGroup(Group.ALL, padPin);
				footprint.addGroup(Group.LEFT, padPin);

				y += pitch;
			}
		}
	};
	public SideEditable getLSide() {
		return this.lSide;
	}

	private final EditableDouble verticalOffset = new EditableDouble();
	public double getVerticalOffset() {
		return this.verticalOffset.doubleValue();
	}
	public EditableDouble getVerticalOffsetEditable() {
		return this.verticalOffset;
	}

	private final EditableDouble horizontalOffset = new EditableDouble();
	public double getHorizontalOffset() {
		return this.horizontalOffset.doubleValue();
	}
	public EditableDouble getHorizontalOffsetEditable() {
		return this.horizontalOffset;
	}

	private final PadEditable epad = new PadEditable();
	public PadEditable getEPad() {
		return this.epad;
	}

	public void fillFootprint(final Footprint footprint) {
		this.number = 1;

		final double halfVOffset = (this.getVerticalOffset() / 2.0D);
		final double halfHOffset = (this.getHorizontalOffset() / 2.0D);
		this.getBSide().fillFootprint(footprint, halfVOffset);
		this.getRSide().fillFootprint(footprint, halfHOffset);
		this.getTSide().fillFootprint(footprint, halfVOffset);
		this.getLSide().fillFootprint(footprint, halfHOffset);

		if (epad.isValid()) {
			final PadPin padPin = new PadPin(0.0D, 0.0D, "EP");
			fillPadPin(padPin, epad, 0.0D, OffsetType.CENTRED, footprint.getOptions());

			footprint.addGroup(Group.ALL, padPin);
			footprint.addGroup(Group.EXPOSEDPAD, padPin);
		}
	}

	private static void fillPadPin(final PadPin padPin, final PadEditable pad, final double angle, final OffsetType offsetType, final OptionsEditable options) {
		final StyleType style = pad.getPadStyle();
		padPin.setPad(true);
		padPin.setPadWidth(pad.getPadWidth());
		padPin.setPadLength(((style != StyleType.SQUARE) ? (pad.getPadLength()) : (pad.getPadWidth())));
		padPin.setRectangle((style == StyleType.RECTANGLE) || (style == StyleType.SQUARE));
		padPin.setAngle(angle);
		padPin.setMaskOffset(options.getMaskOffset());
		padPin.setClearanceOffset(options.getClearanceOffset());
		padPin.setYOffset(offsetType.getOffset(padPin.getPadLength() / 2.0D));
	}

	private int number;
	private String getNextNumber() {
		return Integer.toString(this.number++);
	}
}

