/*
 * Copyright (C) 2008 Robert Fitzsimons
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  Look for the COPYING file in the top
 * level directory.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.cyclerecorder.footprintbuilder.editable;

import org.cyclerecorder.footprintbuilder.data.Footprint;
import org.cyclerecorder.footprintbuilder.data.Group;
import org.cyclerecorder.footprintbuilder.data.PadPin;
import org.cyclerecorder.footprintbuilder.data.OffsetType;
import org.cyclerecorder.footprintbuilder.data.StyleType;
import org.cyclerecorder.footprintbuilder.editable.EditableEnum;
import org.cyclerecorder.footprintbuilder.editable.EditableInteger;
import org.cyclerecorder.footprintbuilder.editable.EditableDouble;

public class QuadEditable {
	private final PadEditable xPad = new PadEditable();
	public PadEditable getXPad() {
		return this.xPad;
	}

	private final PadEditable yPad = new PadEditable();
	public PadEditable getYPad() {
		return this.yPad;
	}

	private final PadEditable ePad = new PadEditable();
	public PadEditable getEPad() {
		return this.ePad;
	}

	private boolean square;
	public boolean isSquare() {
		return this.square;
	}
	public void setSquare(final boolean square) {
		this.square = square;
	}

	private final EditableInteger xCount = new EditableInteger();
	public int getXCount() {
		return this.xCount.intValue();
	}
	public EditableInteger getXCountEditable() {
		return this.xCount;
	}

	private final EditableInteger yCount = new EditableInteger();
	public int getYCount() {
		return this.yCount.intValue();
	}
	public EditableInteger getYCountEditable() {
		return this.yCount;
	}

	private final EditableDouble xPitch = new EditableDouble();
	public double getXPitch() {
		return this.xPitch.doubleValue();
	}
	public EditableDouble getXPitchEditable() {
		return this.xPitch;
	}

	private final EditableDouble yPitch = new EditableDouble();
	public double getYPitch() {
		return this.yPitch.doubleValue();
	}
	public EditableDouble getYPitchEditable() {
		return this.yPitch;
	}

	private final EditableDouble horizontalPitch = new EditableDouble();
	public double getHorizontalPitch() {
		return this.horizontalPitch.doubleValue();
	}
	public EditableDouble getHorizontalPitchEditable() {
		return this.horizontalPitch;
	}

	private final EditableDouble verticalPitch = new EditableDouble();
	public double getVerticalPitch() {
		return this.verticalPitch.doubleValue();
	}
	public EditableDouble getVerticalPitchEditable() {
		return this.verticalPitch;
	}

	private final EditableEnum<OffsetType> offsetType = new EditableEnum<OffsetType>(OffsetType.values(), OffsetType.CENTRED);
	public OffsetType getOffsetType() {
		return this.offsetType.getValue();
	}
	public EditableEnum<OffsetType> getOffsetTypeEditable() {
		return this.offsetType;
	}

	public void fillFootprint(final Footprint footprint) {
		final OptionsEditable options = footprint.getOptions();
		final int xcount = this.getXCount();
		final int ycount = ((!this.isSquare()) ? (this.getYCount()) : (xcount));
		final double xpitch = this.getXPitch();
		final double ypitch = ((!this.isSquare()) ? (this.getYPitch()) : (xpitch));
		final double vpitch = this.getVerticalPitch();
		final double hpitch = ((!this.isSquare()) ? (this.getHorizontalPitch()) : (vpitch));
		final PadEditable xpad = this.getXPad();
		final PadEditable ypad = ((!this.isSquare()) ? (this.getYPad()) : (xpad));
		final PadEditable epad = this.getEPad();
		final OffsetType offsetType = this.getOffsetType();

		int number = 1;
		double x;
		double y;

		x = -((xpitch * (xcount - 1)) / 2.0D);
		y = (vpitch / 2.0D);
		for (int ix = 0; ix < xcount; ix++) {
			final String num = Integer.toString(number++);
			final PadPin padPin = new PadPin(x, y, num);
			fillPadPin(padPin, xpad, 180.0D, offsetType, options);

			footprint.addGroup(Group.ALL, padPin);
			footprint.addGroup(Group.BOTTOM, padPin);
			if ((number - 1) == 1) {
				footprint.addGroup(Group.PINONE, padPin);
			}

			x += xpitch;
		}

		x = (hpitch / 2.0D);
		y = ((ypitch * (ycount - 1)) / 2.0D);
		for (int iy = 0; iy < ycount; iy++) {
			final String num = Integer.toString(number++);
			final PadPin padPin = new PadPin(x, y, num);
			fillPadPin(padPin, ypad, 90.0D, offsetType, options);

			footprint.addGroup(Group.ALL, padPin);
			footprint.addGroup(Group.RIGHT, padPin);

			y -= ypitch;
		}

		x = ((xpitch * (xcount - 1)) / 2.0D);
		y = -(vpitch / 2.0D);
		for (int ix = 0; ix < xcount; ix++) {
			final String num = Integer.toString(number++);
			final PadPin padPin = new PadPin(x, y, num);
			fillPadPin(padPin, xpad, 0.0D, offsetType, options);

			footprint.addGroup(Group.ALL, padPin);
			footprint.addGroup(Group.TOP, padPin);

			x -= xpitch;
		}

		x = -(hpitch / 2.0D);
		y = -((ypitch * (ycount - 1)) / 2.0D);
		for (int iy = 0; iy < ycount; iy++) {
			final String num = Integer.toString(number++);
			final PadPin padPin = new PadPin(x, y, num);
			fillPadPin(padPin, ypad, 270.0D, offsetType, options);

			footprint.addGroup(Group.ALL, padPin);
			footprint.addGroup(Group.LEFT, padPin);

			y += ypitch;
		}

		if (epad.isValid()) {
			final PadPin padPin = new PadPin(0.0D, 0.0D, "EP");
			fillPadPin(padPin, epad, 0.0D, OffsetType.CENTRED, options);

			footprint.addGroup(Group.ALL, padPin);
			footprint.addGroup(Group.EXPOSEDPAD, padPin);
		}
	}

	private static void fillPadPin(final PadPin padPin, final PadEditable pad, final double angle, final OffsetType offsetType, final OptionsEditable options) {
		final StyleType style = pad.getPadStyle();
		padPin.setPad(true);
		padPin.setPadWidth(pad.getPadWidth());
		padPin.setPadLength(((style != StyleType.SQUARE) ? (pad.getPadLength()) : (pad.getPadWidth())));
		padPin.setRectangle((style == StyleType.RECTANGLE) || (style == StyleType.SQUARE));
		padPin.setAngle(angle);
		padPin.setMaskOffset(options.getMaskOffset());
		padPin.setClearanceOffset(options.getClearanceOffset());
		padPin.setYOffset(offsetType.getOffset(padPin.getPadLength() / 2.0D));
	}
}

