/*
 * Copyright (C) 2008 Robert Fitzsimons
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  Look for the COPYING file in the top
 * level directory.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.cyclerecorder.footprintbuilder.editable;

import org.cyclerecorder.footprintbuilder.data.Footprint;
import org.cyclerecorder.footprintbuilder.data.Group;
import org.cyclerecorder.footprintbuilder.data.PadPin;
import org.cyclerecorder.footprintbuilder.data.PinNumberType;
import org.cyclerecorder.footprintbuilder.data.StyleType;
import org.cyclerecorder.footprintbuilder.editable.OptionsEditable;
import org.cyclerecorder.footprintbuilder.editable.EditableInteger;
import org.cyclerecorder.footprintbuilder.editable.EditableDouble;

public class GridEditable {
	private static final char[] ALPHANUMBERS = {
		'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K',
		'L', 'M', 'N', 'P', 'R', 'T', 'U', 'V', 'W', 'Y'
	};

	private final PadEditable pad = new PadEditable();
	public PadEditable getPad() {
		return this.pad;
	}

	private final PinEditable pin = new PinEditable();
	public PinEditable getPin() {
		return this.pin;
	}

	private final EditableInteger xCount = new EditableInteger();
	public int getXCount() {
		return this.xCount.intValue();
	}
	public EditableInteger getXCountEditable() {
		return this.xCount;
	}

	private final EditableInteger yCount = new EditableInteger();
	public int getYCount() {
		return this.yCount.intValue();
	}
	public EditableInteger getYCountEditable() {
		return this.yCount;
	}

	private final EditableDouble xPitch = new EditableDouble();
	public double getXPitch() {
		return this.xPitch.doubleValue();
	}
	public EditableDouble getXPitchEditable() {
		return this.xPitch;
	}

	private final EditableDouble yPitch = new EditableDouble();
	public double getYPitch() {
		return this.yPitch.doubleValue();
	}
	public EditableDouble getYPitchEditable() {
		return this.yPitch;
	}

	private final EditableEnum<PinNumberType> pinNumberType = new EditableEnum<PinNumberType>(PinNumberType.values(), PinNumberType.NUMERIC);
	public PinNumberType getPinNumberType() {
		return this.pinNumberType.getValue();
	}
	public EditableEnum<PinNumberType> getPinNumberTypeEditable() {
		return this.pinNumberType;
	}

	public void fillFootprint(final Footprint footprint) {
		final OptionsEditable options = footprint.getOptions();
		final int xcount = this.getXCount();
		final int ycount = this.getYCount();
		final double xpitch = this.getXPitch();
		final double ypitch = this.getYPitch();

		final boolean isPad = this.getPad().isValid();
		final boolean isPin = this.getPin().isValid();
		final PinNumberType pinNumberType = this.getPinNumberType();

		int number = 1;
		double x;
		double y;

		y = ((ypitch * (ycount - 1)) / 2.0D);
		for (int iy = 0; iy < ycount; iy++) {
			x = -((xpitch * (xcount - 1)) / 2.0D);
			for (int ix = 0; ix < xcount; ix++) {
				final String num = pinNumberType.getPinNumber(number++, ix, iy);
				final PadPin padPin = new PadPin(x, y, num);

				if (isPad) {
					fillPad(padPin, this.pad, options);
				}
				if (isPin) {
					fillPin(padPin, this.pin, options);
				}

				footprint.addGroup(Group.ALL, padPin);
				footprint.addGroup(Group.GRID, padPin);
				if ((number - 1) == 1) {
					footprint.addGroup(Group.PINONE, padPin);
				}

				x += xpitch;
			}
			y -= ypitch;
		}
	}

	private static void fillPad(final PadPin padPin, final PadEditable pad, final OptionsEditable options) {
		final StyleType style = pad.getPadStyle();
		padPin.setPad(true);
		padPin.setPadWidth(pad.getPadWidth());
		padPin.setPadLength(((style == StyleType.SQUARE) || (style == StyleType.CIRCLE)) ? (pad.getPadWidth()) : (pad.getPadLength()));
		padPin.setRectangle((style == StyleType.RECTANGLE) || (style == StyleType.SQUARE));
		padPin.setMaskOffset(options.getMaskOffset());
		padPin.setClearanceOffset(options.getClearanceOffset());
	}
	private static void fillPin(final PadPin padPin, final PinEditable pin, final OptionsEditable options) {
		padPin.setPin(true);
		padPin.setPinDiameter(pin.getPinDiameter());
		padPin.setHoleDiameter(pin.getHoleDiameter());
		padPin.setMaskOffset(options.getMaskOffset());
		padPin.setClearanceOffset(options.getClearanceOffset());
	}
}

