/*
 * Copyright (C) 2008 Robert Fitzsimons
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  Look for the COPYING file in the top
 * level directory.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.cyclerecorder.footprintbuilder.data;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.NavigableSet;
import java.util.TreeSet;

public final class SegmentLine {
	private final Comparator<Segment> comparator = new Comparator<Segment>() {
		public int compare(final Segment o1, final Segment o2) {
			return o1.point.compareTo(o2.point);
		}
	};
	private final TreeSet<Segment> segments = new TreeSet<Segment>(this.comparator);
	private	final double x1;
	private	final double y1;
	private	final double x2;
	private	final double y2;

	public SegmentLine(final double x1, final double y1, final double x2, final double y2) {
		this.x1 = x1;
		this.y1 = y1;
		this.x2 = x2;
		this.y2 = y2;
		this.add(x1, y1, x2, y2);
	}

	public void add(final double x1, final double y1, final double x2, final double y2) {
		final Point p1 = new Point(x1, y1);
		final Point p2 = new Point(x2, y2);
		this.add(p1, p2);
	}
	public void add(final Point p1, final Point p2) {
		Point from = p1;
		Point to = p2;
		if (p1.compareTo(p2) > 0) {
			from = p2;
			to = p1;
		}
		final Segment segFrom = new Segment(from, true);
		final Segment segTo = new Segment(to, false);
		this.apply(segFrom, segTo);
	}

	public void remove(final double x1, final double y1, final double x2, final double y2) {
		final Point p1 = new Point(x1, y1);
		final Point p2 = new Point(x2, y2);
		this.remove(p1, p2);
	}
	public void remove(final Point p1, final Point p2) {
		Point from = p1;
		Point to = p2;
		if (p1.compareTo(p2) > 0) {
			from = p2;
			to = p1;
		}
		final Segment segFrom = new Segment(from, false);
		final Segment segTo = new Segment(to, false);
		this.apply(segFrom, segTo);
	}

	private void apply(final Segment segFrom, final Segment segTo) {
		final NavigableSet<Segment> subSet = this.segments.subSet(segFrom, true, segTo, true);
		if (subSet.isEmpty()) {
			final Segment s = this.segments.floor(segTo);
			if (s != null) {
				segTo.line = s.line;
			}
			this.segments.add(segFrom);
			this.segments.add(segTo);
		} else if (subSet.size() > 0) {
			final Segment s = this.segments.floor(segTo);
			segTo.line = s.line;
			subSet.clear();
			this.segments.add(segFrom);
			this.segments.add(segTo);
		}
	}

	public void clear() {
		this.segments.clear();
	}

	public ArrayList<Line> getLines() {
		final ArrayList<Line> lines = new ArrayList<Line>();
		Point point = null;
		for (final Segment segment : this.segments) {
			if (point != null) {
				lines.add(new Line(point, segment.point));
				point = null;
			}
			if (segment.line) {
				point = segment.point;
			}
		}
		if (point != null) {
			lines.add(new Line(point, point));
			point = null;
		}
		return lines;
	}

	public String toString() {
		return this.segments.toString() + " => " + this.getLines();
	}

	private final static class Segment {
		private final Point point;
		private boolean line;

		private Segment(final Point point, final boolean line) {
			this.point = point;
			this.line = line;
		}

		public String toString() {
			final StringBuilder builder = new StringBuilder();
			builder.append((this.line) ? ("Line") : ("Hole")).append(this.point);
			return builder.toString();
		}
	}

	public void splitLine(final ArrayList<BoundingRectangle> bounds) {
		final boolean horizontal = (y1 == y2);
		final boolean vertical = (x1 == x2);

		if (horizontal && vertical) {
			for (final BoundingRectangle bound : bounds) {
				final double l = bound.getLeft();
				final double r = bound.getRight();
				final double t = bound.getTop();
				final double b = bound.getBottom();

				if ((x1 > l) && (x1 < r) && (y1 > t) && (y1 < b)) {
					this.clear();
				}
			}
		} else if (horizontal) {
			for (final BoundingRectangle bound : bounds) {
				final double l = bound.getLeft();
				final double r = bound.getRight();
				final double t = bound.getTop();
				final double b = bound.getBottom();

				if ((y1 > t) && (y1 < b)) {
					this.remove(new Point(l, y1), new Point(r, y1));
				}
			}
		} else if (vertical) {
			for (final BoundingRectangle bound : bounds) {
				final double l = bound.getLeft();
				final double r = bound.getRight();
				final double t = bound.getTop();
				final double b = bound.getBottom();

				if ((x1 > l) && (x1 < r)) {
					this.remove(new Point(x1, t), new Point(x1, b));
				}
			}
		} else {
			final double _m = (y2 - y1) / (x2 - x1);
			final double _b =  y1 - (x1 * _m);

			for (final BoundingRectangle bound : bounds) {
				final double l = bound.getLeft();
				final double r = bound.getRight();
				final double t = bound.getTop();
				final double b = bound.getBottom();

				if (((x1 < l) && (x2 < l)) || ((x1 > r) && (x2 > r)) || ((y1 < t) && (y2 < t)) || ((y1 > b) && (y2 > b))) {
					continue;
				}

				Point pl = null;
				final double yl = (_m * l) + _b;
				if ((yl >= t) && (yl <= b)) {
					pl = new Point(l, yl);
				}
				Point pr = null;
				final double yr = (_m * r) + _b;
				if ((yr >= t) && (yr <= b)) {
					pr = new Point(r, yr);
				}
				Point pt = null;
				final double xt = (t - _b) / _m;
				if ((xt >= l) && (xt <= r)) {
					pt = new Point(xt, t);
				}
				Point pb = null;
				final double xb = (b - _b) / _m;
				if ((xb >= l) && (xb <= r)) {
					pb = new Point(xb, b);
				}

				if ((pl != null) && (pr != null)) {
					this.remove(pl, pr);
				} else if ((pl != null) && (pt != null)) {
					this.remove(pl, pt);
				} else if ((pl != null) && (pb != null)) {
					this.remove(pl, pb);
				} else if ((pr != null) && (pt != null)) {
					this.remove(pr, pt);
				} else if ((pr != null) && (pb != null)) {
					this.remove(pr, pb);
				} else if ((pt != null) && (pb != null)) {
					this.remove(pt, pb);
				}
			}
		}
	}
}

