/*
 * Copyright (C) 2008 Robert Fitzsimons
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  Look for the COPYING file in the top
 * level directory.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.cyclerecorder.footprintbuilder.data;

import org.cyclerecorder.footprintbuilder.editable.SilkEditable;
import org.cyclerecorder.footprintbuilder.editable.OptionsEditable;
import org.cyclerecorder.footprintbuilder.editable.RectangleEditable;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.IdentityHashMap;

import org.cyclerecorder.footprintbuilder.*;
import org.cyclerecorder.footprintbuilder.editable.*;
import org.cyclerecorder.footprintbuilder.renderer.*;

import java.io.IOException;

public final class Footprint {
	private final ArrayList<SilkLine> silkLines = new ArrayList<SilkLine>();
	private final EnumMap<Group, ArrayList<PadPin>> groupLists = new EnumMap<Group, ArrayList<PadPin>>(Group.class);

	public Footprint(final UnitType unitType, final OptionsEditable options) {
		this.name = "Test";
		this.unitType = unitType;
		this.options = options;
	}

	public String toString() {
		return this.groupLists + " " + this.silkLines;
	}

	private final UnitType unitType;
	public UnitType getUnitType() {
		return this.unitType;
	}

	private final OptionsEditable options;
	public OptionsEditable getOptions() {
		return this.options;
	}

	private String name;
	public String getName() {
		return this.name;
	}
	public void setName(final String name) {
		this.name = name;
	}

	private String description;
	public String getDescription() {
		return this.description;
	}
	public void setDescription(final String description) {
		this.description = description;
	}

	public ArrayList<SilkLine> getSilkLines() {
		return this.silkLines;
	}
	public ArrayList<PadPin> getPadPins() {
		return this.getGroup(Group.ALL);
	}

	private ArrayList<PadPin> getGroup(final Group group) {
		ArrayList<PadPin> groupList = this.groupLists.get(group);
		if (groupList == null) {
			groupList = new ArrayList<PadPin>();
			this.groupLists.put(group, groupList);
		}
		return groupList;
	}
	public void addGroup(final Group group, final PadPin padPin) {
		this.getGroup(group).add(padPin);
	}

	public void addSilkLine(final SilkLine silkLine) {
		this.silkLines.add(silkLine);
	}

	public EnumMap<Group, BoundingRectangle> buildGroupBounds(final double extraOffset) {
		final IdentityHashMap<PadPin, Quadrilateral> padPinBounds = new IdentityHashMap<PadPin, Quadrilateral>();
		final EnumMap<Group, BoundingRectangle> groupBounds = new EnumMap<Group, BoundingRectangle>(Group.class);

		for (final PadPin padPin : this.getPadPins()) {
			final double x = padPin.getX();
			final double y = padPin.getY();
			final double xOffset = padPin.getXOffset();
			final double yOffset = padPin.getYOffset();
			final double angle = padPin.getAngle();
			double halfWidth = 0.0D;
			double halfLength = 0.0D;

			if (padPin.isPad()) {
				halfWidth = (padPin.getPadWidth() / 2.0D);
				halfLength = (padPin.getPadLength() / 2.0D);
			}
			if (padPin.isPin()) {
				final double halfDiameter = (Math.max(padPin.getPinDiameter(), padPin.getHoleDiameter()) / 2.0D);
				halfWidth = Math.max(halfWidth, halfDiameter);
				halfLength = Math.max(halfLength, halfDiameter);
			}

			final double offset = padPin.getMaskOffset() + extraOffset;

			final Quadrilateral quad = new Quadrilateral();

			quad.setP1(-halfWidth + -offset, -halfLength + -offset);
			quad.setP2(halfWidth + offset, -halfLength + -offset);
			quad.setP3(halfWidth + offset, halfLength + offset);
			quad.setP4(-halfWidth + -offset, halfLength + offset);

			quad.translate(xOffset, yOffset);
			quad.rotate(angle);
			quad.translate(x, y);

			padPinBounds.put(padPin, quad);
		}

		for (final Group group : this.groupLists.keySet()) {
			final BoundingRectangle br = new BoundingRectangle();

			for (final PadPin padPin : this.groupLists.get(group)) {
				final Quadrilateral quad = padPinBounds.get(padPin);
				quad.addBounds(br);
			}

			groupBounds.put(group, br);
		}

		return groupBounds;
	}

/*
	private static void drawSilk(final ArrayList<SilkLine> silkLines, final ArrayList<BoundingRectangle> bounds, final EnumMap<Group, BoundingRectangle> groupBounds, final double silkWidth, final Drawable.Silk.Marker silkMarker, final double silkOffset) {
		final BoundingRectangle allBR = groupBounds.get(Group.ALL);
		final double el = allBR.getLeft() + -silkOffset;
		final double er = allBR.getRight() + silkOffset;
		final double et = allBR.getTop() + -silkOffset;
		final double eb = allBR.getBottom() + silkOffset;
		final double markerOffset = (silkWidth * 3.0D);

		silkLines.add(new SilkLine(el, et, er, et, silkWidth));
		silkLines.add(new SilkLine(er, et, er, eb, silkWidth));
		silkLines.add(new SilkLine(er, eb, el, eb, silkWidth));
		silkLines.add(new SilkLine(el, eb, el, et, silkWidth));

		switch (silkMarker) {
		case NONE: break;
		case BEVELED: silkLines.add(new SilkLine(el, eb - markerOffset, el + markerOffset, eb, silkWidth)); break;
		case SQUARE:
			silkLines.add(new SilkLine(el, eb - markerOffset, el + markerOffset, eb - markerOffset, silkWidth));
			silkLines.add(new SilkLine(el + markerOffset, eb - markerOffset, el + markerOffset, eb, silkWidth));
			break;
		case CUP:
			silkLines.add(new SilkLine(el, 0.0D + markerOffset, el + markerOffset, 0.0D + markerOffset, silkWidth));
			silkLines.add(new SilkLine(el + markerOffset, 0.0D + markerOffset, el + markerOffset, 0.0D - markerOffset, silkWidth));
			silkLines.add(new SilkLine(el, 0.0D - markerOffset, el + markerOffset, 0.0D - markerOffset, silkWidth));
			break;
		case BAND:
			{
				final BoundingRectangle pinOneBR = groupBounds.get(Group.PINONE);
				final double top = pinOneBR.getTop();
				silkLines.add(new SilkLine(el, top, er, top, silkWidth));
			}
			break;
		case AROUNDPINONE:
			{
				final BoundingRectangle pinOneBR = groupBounds.get(Group.PINONE);
				final double left = pinOneBR.getLeft();
				final double right = pinOneBR.getRight();
				final double top = pinOneBR.getTop();
				final double bottom = pinOneBR.getBottom();
				silkLines.add(new SilkLine(left, top, right, top, silkWidth));
				silkLines.add(new SilkLine(right, top, right, bottom, silkWidth));
				silkLines.add(new SilkLine(right, bottom, left, bottom, silkWidth));
				silkLines.add(new SilkLine(left, bottom, left, top, silkWidth));
			}
			break;
		case INSIDE_DOT: silkLines.add(new SilkLine(el + markerOffset, eb - markerOffset, el + markerOffset, eb - markerOffset, (silkWidth * 2.0D))); break;
		case OUTSIDE_DOT: silkLines.add(new SilkLine(el - markerOffset, eb + markerOffset, el - markerOffset, eb + markerOffset, (silkWidth * 2.0D))); break;
		case INSIDE_TICK: silkLines.add(new SilkLine(el, eb, el + markerOffset, eb - markerOffset, silkWidth)); break;
		case OUTSIDE_TICK: silkLines.add(new SilkLine(el, eb, el - markerOffset, eb + markerOffset, silkWidth)); break;
		case INSIDE_LINE: silkLines.add(new SilkLine(el, eb - markerOffset, er, eb - markerOffset, silkWidth)); break;
		case OUTSIDE_LINE: silkLines.add(new SilkLine(el, eb + markerOffset, er, eb + markerOffset, silkWidth)); break;
		}
	}

	private static final ArrayList<SilkLine> splitLine(final SilkLine silkLine, final ArrayList<BoundingRectangle> bounds) {
		final double x1 = silkLine.getX1();
		final double y1 = silkLine.getY1();
		final double x2 = silkLine.getX2();
		final double y2 = silkLine.getY2();
		final SegmentLine segmentLine = new SegmentLine(x1, y1, x2, y2);
		final boolean horizontal = (y1 == y2);
		final boolean vertical = (x1 == x2);

		if (horizontal && vertical) {
			for (final BoundingRectangle bound : bounds) {
				final double l = bound.getLeft();
				final double r = bound.getRight();
				final double t = bound.getTop();
				final double b = bound.getBottom();

				if ((x1 > l) && (x1 < r) && (y1 > t) && (y1 < b)) {
					segmentLine.clear();
				}
			}
		} else if (horizontal) {
			for (final BoundingRectangle bound : bounds) {
				final double l = bound.getLeft();
				final double r = bound.getRight();
				final double t = bound.getTop();
				final double b = bound.getBottom();

				if ((y1 > t) && (y1 < b)) {
					segmentLine.remove(new Point(l, y1), new Point(r, y1));
				}
			}
		} else if (vertical) {
			for (final BoundingRectangle bound : bounds) {
				final double l = bound.getLeft();
				final double r = bound.getRight();
				final double t = bound.getTop();
				final double b = bound.getBottom();

				if ((x1 > l) && (x1 < r)) {
					segmentLine.remove(new Point(x1, t), new Point(x1, b));
				}
			}
		} else {
			final double _m = (y2 - y1) / (x2 - x1);
			final double _b =  y1 - (x1 * _m);

			for (final BoundingRectangle bound : bounds) {
				final double l = bound.getLeft();
				final double r = bound.getRight();
				final double t = bound.getTop();
				final double b = bound.getBottom();

				if (((x1 < l) && (x2 < l)) || ((x1 > r) && (x2 > r)) || ((y1 < t) && (y2 < t)) || ((y1 > b) && (y2 > b))) {
					continue;
				}

				Point pl = null;
				final double yl = (_m * l) + _b;
				if ((yl >= t) && (yl <= b)) {
					pl = new Point(l, yl);
				}
				Point pr = null;
				final double yr = (_m * r) + _b;
				if ((yr >= t) && (yr <= b)) {
					pr = new Point(r, yr);
				}
				Point pt = null;
				final double xt = (t - _b) / _m;
				if ((xt >= l) && (xt <= r)) {
					pt = new Point(xt, t);
				}
				Point pb = null;
				final double xb = (b - _b) / _m;
				if ((xb >= l) && (xb <= r)) {
					pb = new Point(xb, b);
				}

				if ((pl != null) && (pr != null)) {
					segmentLine.remove(pl, pr);
				} else if ((pl != null) && (pt != null)) {
					segmentLine.remove(pl, pt);
				} else if ((pl != null) && (pb != null)) {
					segmentLine.remove(pl, pb);
				} else if ((pr != null) && (pt != null)) {
					segmentLine.remove(pr, pt);
				} else if ((pr != null) && (pb != null)) {
					segmentLine.remove(pr, pb);
				} else if ((pt != null) && (pb != null)) {
					segmentLine.remove(pt, pb);
				}
			}

		}

		final ArrayList<Line> newLines = segmentLine.getLines();
		final ArrayList<SilkLine> newSilkLines = new ArrayList<SilkLine>();
		for (final Line line : newLines) {
			newSilkLines.add(new SilkLine(line.getX1(), line.getY1(), line.getX2(), line.getY2(), silkLine.getThickness()));
		}
		return newSilkLines;
	}
*/
	public static void main(final String[] args) throws IOException {
/*
		final StringBuilder builder = new StringBuilder();
		final ArrayList<PadPin> list = new ArrayList<PadPin>();

		final PadPin pp1 = new PadPin();
		pp1.setPad(true);
		pp1.setPadWidth(250.0D);
		pp1.setPadLength(500.0D);
		list.add(pp1);

		final PadPin pp2 = new PadPin();
		pp2.setPad(true);
		pp2.setPadWidth(250.0D);
		pp2.setPadLength(500.0D);
		pp2.setXOffset(100.0D);
		pp2.setYOffset(50.0D);
		pp2.setX(500.0D);
		list.add(pp2);

		final PadPin pp3 = new PadPin();
		pp3.setPad(true);
		pp3.setPadWidth(250.0D);
		pp3.setPadLength(500.0D);
		pp3.setXOffset(100.0D);
		pp3.setYOffset(50.0D);
		pp3.setAngle(30.0D);
		pp3.setX(1000.0D);
		list.add(pp3);

		System.out.println(builder);
*/

		final RectangleEditable qe = new RectangleEditable();

		qe.getVerticalOffsetEditable().setValue(2.50D);

		qe.getTSide().getCountEditable().setValue(1);
		qe.getTSide().getPitchEditable().setValue(0.00D);
		qe.getTSide().getOffsetTypeEditable().setValue(OffsetType.INSIDE);
		qe.getTSide().getPad().getPadWidthEditable().setValue(0.50D);
		qe.getTSide().getPad().getPadLengthEditable().setValue(0.45D);

		qe.getBSide().getCountEditable().setValue(2);
		qe.getBSide().getPitchEditable().setValue(1.90D);
		qe.getBSide().getOffsetTypeEditable().setValue(OffsetType.INSIDE);
		qe.getBSide().getPad().getPadWidthEditable().setValue(0.50D);
		qe.getBSide().getPad().getPadLengthEditable().setValue(0.45D);

		final SilkEditable se = new SilkEditable();
		se.getSilkWidthEditable().setValue(3.00D);
		se.getSilkLengthEditable().setValue(1.20D);

		final OptionsEditable oe = new OptionsEditable();
		oe.getMaskOffsetEditable().setValue(0.05D);
		oe.getClearanceOffsetEditable().setValue(0.05D);
		oe.getSilkLineWidthEditable().setValue(0.10D);

		final Footprint footprint = new Footprint(UnitType.MM, oe);
		qe.fillFootprint(footprint);
		se.fillFootprint(footprint);

		final StringBuilder builder = new StringBuilder();
		final PcbRenderer pcbRenderer = new PcbRenderer();
		pcbRenderer.render(builder, footprint);
		System.out.println(builder);

/*
		final BufferedImage image = new BufferedImage(800, 600, BufferedImage.TYPE_INT_ARGB);
		final Renderer<Graphics2D> g2Renderer = new Graphics2DRenderer(UnitType.MIL, "TEST");
		final Graphics2D g2 = image.createGraphics();
		g2Renderer.render(g2, padPins, silkLines);
		ImageIO.write(image, "png", new File("test.png"));
*/
	}
}

