/*
 * footprintbuilder - An interactive footprint design tool for EDA tools.
 * Copyright (C) 2008, 2009 Robert Fitzsimons
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  Look for the COPYING file in the top
 * level directory.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.cyclerecorder.footprintbuilder;

import java.awt.BorderLayout;
import java.awt.EventQueue;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import javax.imageio.ImageIO;
import javax.swing.AbstractAction;
import javax.swing.ButtonGroup;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.WindowConstants;
import javax.swing.filechooser.FileNameExtensionFilter;

import org.cyclerecorder.footprintbuilder.data.*;
import org.cyclerecorder.footprintbuilder.editor.*;
import org.cyclerecorder.footprintbuilder.renderer.*;

public final class Main {
	private Main() {}

	public static void main(final String[] args) {
		final JFrame frame = new JFrame("Footprint Builder");
		frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
		frame.setSize(800, 600);

		final FootprintEditor footprintEditor = new FootprintEditor();
		final FootprintView footprintView = new FootprintView(footprintEditor);
		footprintEditor.setFootprintView(footprintView);

		frame.add(footprintView, BorderLayout.CENTER);
		frame.add(footprintEditor, BorderLayout.WEST);

		final JMenuBar menuBar = new JMenuBar();

		final JMenu exportMenu = new JMenu("Export");
		exportMenu.add(new AbstractAction("pcb dump") {
			public void actionPerformed(final ActionEvent ae) {
				final AppendableRenderer ar = new PcbRenderer();
				try {
					final StringBuilder builder = new StringBuilder();
					ar.render(builder, footprintEditor.getFootprint());
					System.out.println(builder);
				} catch (final IOException ioe) {
					System.err.println(ioe);
				}
			}
		});
		exportMenu.addSeparator();
		exportMenu.add(new FileOutputAction("pcb", footprintEditor, new PcbRenderer(), "fp"));
		exportMenu.add(new FileOutputAction("kicad", footprintEditor, new KicadRenderer(), "emp"));
/*
		exportMenu.addSeparator();
		final JMenu imageMenu = new JMenu("Image");
		for (final String mime : ImageIO.getWriterFormatNames()) {
			imageMenu.add(new JMenuItem(mime));
		}
		exportMenu.add(imageMenu);
*/
		menuBar.add(exportMenu);

		final JMenu unitsMenu = new JMenu("Units");
		final ButtonGroup unitsGroup = new ButtonGroup();
		for (final UnitType unitType : UnitType.values()) {
			final JRadioButtonMenuItem unit = new JRadioButtonMenuItem(new AbstractAction(unitType.toString()) {
				public void actionPerformed(final ActionEvent ae) {
					footprintEditor.setUnitType(unitType);
				}
			});
			unitsGroup.add(unit);
			unitsMenu.add(unit);
		}
		menuBar.add(unitsMenu);

		final JMenu packagesMenu = new JMenu("Packages");
		final ButtonGroup packagesGroup = new ButtonGroup();
		for (final PackageType packageType : PackageType.values()) {
			final JRadioButtonMenuItem pack = new JRadioButtonMenuItem(new AbstractAction(packageType.toString()) {
				public void actionPerformed(final ActionEvent ae) {
					footprintEditor.setPackageType(packageType);
					frame.validate();
				}
			});
			packagesGroup.add(pack);
			packagesMenu.add(pack);
		}
		menuBar.add(packagesMenu);

		frame.setJMenuBar(menuBar);

		frame.setVisible(true);

		return;
	}

	private static final class FileOutputAction extends AbstractAction implements Runnable {
		private final FootprintEditor footprintEditor;
		private final AppendableRenderer renderer;
		private final FileNameExtensionFilter filter;
		private final String extension;
	
		public FileOutputAction(final String name, final FootprintEditor footprintEditor, final AppendableRenderer renderer, final String extension) {
			super(name);
			this.footprintEditor = footprintEditor;
			this.renderer = renderer;
			this.filter = new FileNameExtensionFilter(name, extension);
			this.extension = extension;
		}

		public void actionPerformed(final ActionEvent ae) {
			EventQueue.invokeLater(this);
		}

		public void run() {
			final Footprint footprint = this.footprintEditor.getFootprint();
			final JFileChooser chooser = new JFileChooser(".");
			chooser.setFileFilter(this.filter);
			chooser.setSelectedFile(new File(footprint.getName() + "." + this.extension));

			try {
				while (chooser.showSaveDialog(null) == JFileChooser.APPROVE_OPTION) {
					final File file = chooser.getSelectedFile();
					if (file.exists()) {
						if (JOptionPane.showConfirmDialog(null, ("Overwrite file? " + file)) != JOptionPane.YES_OPTION) {
							continue;
						}
					}
					final Writer writer = new FileWriter(file);
					this.renderer.render(writer, footprint);
					writer.close();
					break;
				}
			} catch (final IOException ioe) {
				System.err.println(ioe);
			}
		}
	}
}

